package easik.sketch.util.Export.Constraints;

import java.util.LinkedList;

import easik.sketch.util.Export.ExportConstants;
import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportPath;


/**
 * A class to create a pullback constraint object, including all necessary information gleaned 
 * from the XML file.  Should not be used for any other purpose than database exportation, as it 
 * contains no graphical information.
 * 
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-08-28 Kevin Green
 *
 */
public class ExportPullback extends ExportConstraint {
	
	/**
	 * The name of the delete procedure for the pullback
	 */
	private final static String PULL_DEL_PROC_NAME = "pullDel";
	/**
	 * The name of the after insert procedure for the pullback
	 */
	private final static String PULL_A_INS_PROC_NAME = "pullAIns";
	
	/**
	 * The codomain of the paths in the pullback
	 */
	private String _targetTable;
	/**
	 * Constructor calling the constructor of the ExportConstraint superclass
	 * @param id A unique id to identify this constraint from all others
	 */
	public ExportPullback(int id) {
		super();
		_id = id;
	}
	
	/**
	 * Sets the base table of this constraint.  This base table is the domain of the two paths which 
	 * form this constraint.
	 */
	public void setBaseTable(){
		_baseTable = _paths.get(0).getDomain();
	}
	
	/**
	 * Sets the target table of this constraint.  This target table is the codomain of the two paths
	 * which form this constraint.
	 */
	public void setTargetTable(){
		_targetTable = _paths.get(1).getCoDomain();
	}
	
	/**
	 * Sets all constraint strings for this constraint.
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setConstraintStrings(){
		setPullbackInsertProcs();
		setAfterInsertProcs();
		setBeforeDeleteProcs();
	}
	
	/**
	 * Sets the procedure to be called <b>BEFORE INSERT</b> on the pull back table (that is, the base table) of this
	 * constraint.  This procedure ensures that any insert into the pullback table conforms to the pullback conditions.
	 *
	 * @since 2007-07-11 Vera Ranieri
	 */
	private void setPullbackInsertProcs(){
		String proc = _db_name + "_" + PULL_A_INS_PROC_NAME + _baseTable + _id;
		_a_i_tables.put(_baseTable, proc);
		
		proc = ExportConstants.CREATE_PROC + proc + ExportConstants.PROC_PARAM;
		proc += ExportConstants.BEGIN;
		proc += "DECLARE var0, var2 INT; \n";
		proc += getBeforeIfStatement();
		proc += getTestForDuplicateStatement();
		proc += ExportConstants.END;
		_procedureStrings.add(proc);
	}
	
	/**
	 * Gets the statements to determine whether the entry is a valid pullback entry.  If this is not the case, the 
	 * procedure will fail
	 * @return The SQL statement to test for the pull back condition
	 * @since 2006-07-14 Vera Ranieri
	 */
	private String getBeforeIfStatement(){
		//TODO: make functional.
		String stmt = getSelectIntoStatement(_paths.get(1), 0);
		stmt += getSelectIntoStatement(_paths.get(3), 2);
		stmt += "IF NOT (var0 <=> var2) \n"
				+ "THEN call Fail('Invalid Entry into Pullback constraint'); \n"
				+ "END IF; \n";
		return stmt;
	}
	
	/**
	 * Creates an <b>INSERT INTO ... SELECT</b> statement to insert into tables along a path
	 * @param path The path for which this insert statement is being called.
	 * @param p The index of the opposite path in the pullback
	 * @return The string of the insert into the pullback table
	 * @since 2006-07-14 Vera Ranieri
	 */
	private String getSelectIntoStatement(ExportPath path, int p){
		LinkedList<ExportEdge> edges = _paths.get(p).getEdges();
		String tables = ExportPath.getAllTablesString(path);
		String conditions = _baseTable +"." +_baseTable +ExportConstants.ID  +"=id " + ExportPath.getConditions(path);
		String source, target;
		for(ExportEdge e: edges){
			
			source = e.getSource();
			target = e.getTarget();
			tables = source + ", " + tables;
			conditions = source +"." +target + ExportConstants.ID + "=" + target+"."+ target + ExportConstants.ID 
						+ "AND " + conditions;
		}
		ExportEdge e = path.getEdges().getLast();
		String stmt = "SELECT " + e.getTarget() +"." + e.getTarget() + ExportConstants.ID +"INTO var" + p + " FROM "
						+ tables + "WHERE " + conditions + "; \n";
		return stmt;
	}
	
	/**
	 * Gets the string that determines whether the pullback has a duplicate entry.
	 * @return The string of this test statement, formatted for SQL
	 * @since 2006-07-14 Vera Ranieri
	 */
	private String getTestForDuplicateStatement(){
		
		String column1 = _paths.get(0).getEdges().getFirst().getTarget() +ExportConstants.ID ;
		String column2 = _paths.get(2).getEdges().getFirst().getTarget() + ExportConstants.ID;
		String cond = column1 + "= (SELECT " + column1  +" FROM " + _baseTable+ " WHERE "  + _baseTable 
						+ExportConstants.ID +"=id) AND " + column2 + "=(SELECT " + column2 +" FROM " + _baseTable +
						" WHERE " + _baseTable +ExportConstants.ID + "=id) ";
		
		String stmt = "IF ( (SELECT COUNT(*) FROM " + _baseTable +" WHERE " + cond
						+ ") > 1 ) \n";
		stmt += "THEN call fail('Duplicate entry in pullback table.'); \n";
		stmt += "END IF; \n";
		return stmt;
	}
	/**
	 * Sets the procedures to be called <b>AFTER INSERT</b> on the tables which form the codomains of the first and third
	 * paths.
	 * 
	 * @since 006-07-07 Vera Ranieri
	 */
	private void setAfterInsertProcs(){
		setAIForTable(_paths.get(1), (_paths.get(3)));
		setAIForTable(_paths.get(3), (_paths.get(1)));
	}
	
	/**
	 * Sets <b>AFTER INSERT</b> procedure for the table which is the domain of path mainP.
	 * 
	 * @param mainP The path which has the table as its domain for which we want to set the procedure.
	 * @param subP The opposite path which has the same codomain as mainP
	 * @since 2006-07-07 Vera Ranieri
	 */
	private void setAIForTable(ExportPath mainP, ExportPath subP ){
		String proc = _db_name + "_" + PULL_A_INS_PROC_NAME + mainP.getDomain()+_id;
		_a_i_tables.put(mainP.getDomain(), proc);
		
		proc = ExportConstants.CREATE_PROC + proc + ExportConstants.PROC_PARAM;
		proc += ExportConstants.BEGIN;
		
		proc += getIfStatement(mainP, subP);
		
		proc += ExportConstants.END;
		
		_procedureStrings.add(proc);
	}
	
	/**
	 * Gets the <b>IF THEN</b> SQL statement for this procedure.  This <b>IF</b> statement checks whether the path leading
	 * from the opposing path contains elements which lead to the same element as the path given, in the target table.
	 * <br>
	 * If this case is determined to be true, a second <b>IF</b> statement determines whether the base table already contains elements 
	 * which link to these elements.  If it does not, elements are added along the path leading to the opposing path table.
	 * <br>
	 * Elements are then added along the path into the original path table.  Finally, the required elements are added to the
	 * pullback (base) table.
	 * <br>
	 * Should the inital case be determined to be false, then nothing will occur when the SQL statement is executed.
	 * 
	 * 
	 * @param main The path for which the procedure is being called
	 * @param alt The path opposing path for this procedure
	 * @return The string of the <b>IF</b> statement.  This string is the body of the procedure.
	 * @since 2006-07-10 Vera Ranieri
	 */
	private String getIfStatement(ExportPath main, ExportPath alt){
		String tables = ", " +ExportPath.getAllTablesString(main);
		LinkedList<ExportEdge> edges = alt.getEdges();
		for(ExportEdge e: edges){
			tables = e.getSource() + tables;
		}
		
		String stmt = "IF( (SELECT COUNT("+alt.getDomain()+"." +alt.getDomain() + ExportConstants.ID
						+") FROM " + tables +" WHERE " + getAltPathConditions(alt, main)+
						") != 0 )\n THEN \n";
		
		stmt += getSecondIfStatement(main, alt);
		
		
		stmt += getFinalInsertStatements(main);
		
		stmt += "END IF; \n";
		return stmt;
	}
	
	/**
	 * Gets the alternate path conditions for testing whether a pullback condition is formed.  This is acheived by determining
	 * the conditions needed to determine if a match exists in the target table of the pullback.  
	 * <br>
	 * Initially, the path is followed from the main path (i.e. the path for which this procedure is being called) to the 
	 * target table.  The path is then followed from the alternate path (i.e. the path which has as its codomain the
	 * same codomain as the main path) to see if a match occurs.
	 * 
	 * @param alt The path which does not contain the table for which the procedure is being called.
	 * @param main The path which contains the table to which the procedure is being called.
	 * @return The string of the conditions to determine whether a match occurs.
	 * @since 2007-07-10 Vera Ranieri
	 */
	private String getAltPathConditions(ExportPath alt, ExportPath main){
	
		LinkedList<ExportEdge> edges = main.getEdges();
		String stmt = edges.getFirst().getSource()+"."+edges.getFirst().getSource()+ ExportConstants.ID+"= id AND ";
		String source, target;
		for(ExportEdge e : edges){
			source = e.getSource();
			target = e.getTarget();
			stmt += source +"."+ target + ExportConstants.ID +"="+target+"." + target+ExportConstants.ID +"AND " ;
		}
		
		edges = alt.getEdges();
		for(ExportEdge e: edges){
			source = e.getSource();
			target = e.getTarget();
			stmt += source+ "."+target+ExportConstants.ID+"=" +target+"."+target+ExportConstants.ID + "AND ";
			
		}
		stmt = stmt.substring(0, stmt.length()-4);
		return stmt;
	}
	
	/**
	 * Gets the main path conditions to determine whether the pullback table contains elements along the path for
	 * every element in the alternate table which leads to the same element as the main table.  
	 * <br>
	 * Should these elements not be yet present in the pullback table, elements are added in the path for every element 
	 * in the alternate table.
	 * <br>
	 * This second <b>IF</b> statement ensures that elements along the opposite path of the pullback are not added
	 * until a match is formed in the pullback.  If these elements have already been added (as by an earlier addition into 
	 * the domain of the main path) nothing will occur when the SQL statement is executed.
	 * 
	 * @param alt The path which does not contain the table for which the procedure is being called.
	 * @param main The path which contains the table to which the procedure is being called.
	 * @return The string of the second <b>IF</b> statement, described above.
	 * @since 2006-07-10 Vera Ranieri
	 */
	private String getSecondIfStatement(ExportPath main, ExportPath alt){
		
		String tables = ExportPath.getAllTablesString(alt);
		LinkedList<ExportEdge> edges = main.getEdges();
		for(ExportEdge e: edges){
			tables = e.getSource() +", " + tables;
		}
		//Determine the path which leads to the alternate path, and get its edges.
		
		if(main.equals(_paths.get(1)))
			edges = _paths.get(2).getEdges();
		
		else
			edges = _paths.get(0).getEdges();
		
		//Checks to see if there are no inserts necessary along the path.
		if(edges.size()==1)
			return "";
		
		String conditions = getAltPathConditions(alt, main);
		
		String source, target;
		
		//TODO: rewrite for better functionality.
		String ins = "";
		String temp = "";
		ExportEdge e;
		for(int i = edges.size() - 1; i >= 0; i--){
			e = edges.get(i);
			source = e.getSource();
			target = e.getTarget();
			
			
			temp = "INSERT INTO " + source +" ("+ target + ExportConstants.ID +") " + "SELECT "+ 
				target+"." + target + ExportConstants.ID+ "FROM " + tables +  
				" WHERE " + conditions +"; \n";
			
			ins += temp;
			tables = source + ", " + tables;
			conditions = source + "." + target + ExportConstants.ID + "=" +
						target + "."+ target+ExportConstants.ID + "AND "+ conditions;
					
		}
		
		String stmt = "IF( (SELECT COUNT("+_paths.get(0).getDomain()+"."+_paths.get(0).getDomain()+ ExportConstants.ID 
		+") FROM " + tables+ " WHERE ";
		
		stmt += conditions + ") = 0 )\n THEN \n";
		
		ins = ins.substring(0, ins.length() -temp.length());
		
		
		
		stmt += ins;
		
		stmt += "END IF; \n";
		return stmt;
	}
	
	/**
	 * Determines the <b>INSERT</b> statements necessary for the procedure called <b>AFTER INSERT</b> on the domain of 
	 * <it>path</it>.  This includes inserting elements into any tables which are on the path leading from the base
	 * table of the pullback to the domain of <it>path</it>.
	 * <br>
	 * If inserts are necessary along that path, they are made prior to any inserts made into the pullback table.
	 * 
	 * @param path The path which has, as its domain, the table for which the procedure is being called.
	 * @return The string of the <b>INSERT</b> statements.
	 * @since 2007-07-11 Vera Ranieri
	 */
	private String getFinalInsertStatements(ExportPath path){
		String ins = "";
		
		//Determine the path which leads to the path, and get its edges
		LinkedList<ExportEdge> edges;
		if(path.equals(_paths.get(1)))
			edges = _paths.get(0).getEdges();
		
		else
			edges = _paths.get(2).getEdges();
		
		
		String source, target; 
		String conditions = ExportPath.getDelConditions(path);
		String temp = "";
		ExportEdge e;
		String tables = ExportPath.getAllTablesString(path);
		for(int i = edges.size() - 1; i > 0; i--){
			e = edges.get(i);
			source = e.getSource();
			target = e.getTarget();
			
			
			
			temp = "INSERT INTO " + source +" ("+ target + ExportConstants.ID +") " + "SELECT "+ 
				target+"." + target + ExportConstants.ID+ "FROM " + tables + "WHERE " + 
				conditions +"; \n";
			
			ins += temp;
			tables = source + ", " + tables;
			conditions += "AND "+ source +"." + target +ExportConstants.ID +"="+ target +"." + target + ExportConstants.ID;
		}
		
		

		String targetA = _paths.get(0).getEdges().getFirst().getTarget();
		String targetB = _paths.get(2).getEdges().getFirst().getTarget();
		ins += "INSERT INTO "+ _baseTable +" (" +targetA+ExportConstants.ID 
					+ ", "+ targetB + ExportConstants.ID + 
					") " + "SELECT " +
					targetA +"." +targetA+ ExportConstants.ID + ", " +
					targetB +"." +targetB + ExportConstants.ID + 
					"FROM " + getAllTablesInPullback() +
					getSelectStatement(path);
		return ins;
	}
	/**
	 * Determines the conditions required for an insert into the pullback table <b>AFTER INSERT</b>
	 * into the table for which the procedure is being called.
	 * <br>
	 * This table is the domain of path <it>p</it>.  It is noted so that the reference to the parameter <it>id</it>
	 * can be made within the conditions stated.
	 * 
	 * @param p The path for which the select statement must be determined.
	 * @return The string of the select statement
	 * @since 2006-07-07 Vera Ranieri
	 */
	private String getSelectStatement(ExportPath p){
		
		String select = p.getDomain()+"." + p.getDomain()+ExportConstants.ID + "=id ";
		
		String temp = 	ExportPath.getConditions(_paths.get(1)) +
						ExportPath.getConditions(_paths.get(3)) +
						getSubConditions(_paths.get(0)) +
						getSubConditions(_paths.get(2));
		
		if(p.equals(_paths.get(0))){
			select += temp;
		}
		else{
			select = temp.substring(3) +"AND " + select;
		}
		
		select = "WHERE " + select +"; \n";
		return select;
	}
	
	/**
	 * Gets the conditions for the insert, given the path, minus the initial conditions.  Separate from the 
	 * <it>getCondtions</it> static method in ExportPath to ensure that initial conditions are not included.
	 * @param path The path for which we need the conditions.
	 * @return The string of the condtions.
	 * @since 2006-07-07 Vera Ranieri
	 */
	private String getSubConditions (ExportPath path){
		String conditions, source, target;
		LinkedList<ExportEdge> edges = path.getEdges();
		
		conditions = "";
		ExportEdge e;
		
		for(int i = 1; i< edges.size(); i++){
			e = edges.get(i);
			source = e.getSource();
			target = e.getTarget(); 
			conditions += "AND " + source+ "." +target +ExportConstants.ID +"=" +
							target+"."+target+ExportConstants.ID;
		}
		
		return conditions;
	}
	/**
	 * Sets the procedure to be called <b>BEFORE DELETE</b> on the domain of the pullback constraint.
	 * 
	 * @since 2006-07-04 Vera Ranieri
	 *
	 */
	private void setBeforeDeleteProcs(){
		String proc = _db_name + "_" + PULL_DEL_PROC_NAME +_baseTable+_id;
		_b_d_tables.put(_baseTable, proc);
		
		proc = ExportConstants.CREATE_PROC + proc + ExportConstants.PROC_PARAM;
		proc += ExportConstants.BEGIN;
		
		proc += getDeleteString();
		
		proc += ExportConstants.END;
		
		_procedureStrings.add(proc);
	}
	
	/**
	 * Gets the delete string for SQL deleting from the codomain of the pullback constraint, the element which is the codomain
	 * of the element being deleted from the domain.
	 * 
	 * @return The string formatted for SQL.
	 * @since 2006-07-04 Vera Ranieri
	 */
	private String getDeleteString(){
		
		String delete = "DELETE FROM "+ _targetTable + " USING ";
		
		
		delete += getAllTablesInPullback() +", " + _baseTable
					+" \nWHERE " 
					+ ExportPath.getDelConditions((ExportPath)_paths.get(0)) 
					+ ExportPath.getConditions((ExportPath)_paths.get(1)) 
					+ ExportPath.getConditions((ExportPath)_paths.get(2)) 
					+ ExportPath.getConditions((ExportPath)_paths.get(3))
					+ "; \n";
		return delete;
	}
	
	/**
	 * Returns a list of all names of tables in the pullback, separated by commas.
	 * @return The string of all the tables.
	 * @since 2007-07-11 Vera Ranieri
	 */
	private String getAllTablesInPullback(){
		
		String tables = "";
		
		String target = "";
		LinkedList<ExportEdge> edges;
		for(ExportPath path : _paths){
			edges = path.getEdges();
			for(ExportEdge e: edges){
				target = e.getTarget();
				tables +=  target +", ";
			}
		}
		
		tables = tables.substring(0, tables.length() - target.length() - 4) +" ";
		
		return tables;
	}

}
